/*
 * This source code has been downloaded from
 * http://eegeek.net/content/view/27/32/1/4/ on 15.11.2007
 * where it is/was made available under the
 * Creative Commons Attribution-Noncommercial-Share Alike 2.5 License.
 */

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Windows.Forms;
using System.Collections;

namespace HEXReader
{
    public class VirtualPIC
    {
        public byte[] ProgMem;
        public byte[] EEPROM;
        public byte[] IDlocs;
        public byte[] ConfigWords;
        public byte[] DevID;
        public int ProgMemCapacity, EEPROMCapacity, ConfigWordsNumBytes;
        public ChipType Chip;

        public enum ChipType
        {
            PIC18F4550
        }

        /// <summary>
        /// Initializes the PIC with no data
        /// </summary>
        public VirtualPIC(ChipType chip)
        { 
            switch(chip)
            {
                case ChipType.PIC18F4550:
                default:
                    ProgMemCapacity = 0x8000;
                    EEPROMCapacity = 256;
                    ConfigWordsNumBytes = 14; 
                    break;
            }
        }

        /// <summary>
        /// Initializes the PIC with the given data
        /// </summary>
        /// <param name="chip">The type of PIC targeted</param>
        /// <param name="progmem">Array representing Program memory</param>
        /// <param name="eeprom">Array representing the EEPROM memory</param>
        /// <param name="idlocations">Array representing the ID locations</param>
        /// <param name="configwords">Array representing the Config words</param>
        public VirtualPIC(ChipType chip, byte[] progmem, byte[] eeprom, byte[] idlocations,byte[] configwords):this(chip)
        {
            ProgMem = progmem;
            EEPROM = eeprom;
            IDlocs = idlocations;
            ConfigWords = configwords;
        }

        private string ChunkDisplayArray(ref byte[] array)
        {
            bool emptyline = true;
            string thestring = "";
            byte[] dataline = new byte[16];
            for (int i = 0; i < array.Length; i += 16)
            {
                emptyline = true;
                for (int j = 0; (j < 16); j++)
                {
                    if (i + j < array.Length)
                    {
                        dataline[j] = array[i + j];
                        if (array[i + j] != 0xFF)
                            emptyline = false;
                    }
                    else
                        dataline[j] = 0xFF;
                }
                if (!emptyline)
                {
                    thestring += "[" + HexEncoding.ToString(new byte[] { (byte)((i / 0x100) % 0x100), (byte)(i % 0x100) }) + "]";
                    thestring += HexEncoding.ToString(dataline, true) + "\r\n";
                }
            }
            return thestring;
        }

        public override string ToString()
        {
            string thestring = "";

            thestring += "Program Memory:\r\n\r\n";
            thestring += "      00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F\r\n";
            thestring += ChunkDisplayArray(ref ProgMem);

            thestring += "\r\nEEPROM:\r\n\r\n";
            if (EEPROM != null)
                thestring += ChunkDisplayArray(ref EEPROM);

            thestring += "\r\nID Locations:\r\n\r\n";
            if (IDlocs != null)
            {
                thestring += HexEncoding.ToString(IDlocs, true);
            }

            thestring += "\r\nConfig Words:\r\n\r\n";
            if (ConfigWords != null)
            {
                thestring += HexEncoding.ToString(ConfigWords, true);
            }

            return thestring;
        }
        
    }

    public class HexInterpreter
    {
        private byte AddressMSB = 0;
        int IntAddress;
        public bool IsValid = true;
        string FileName;
        StreamReader hexread;
        StreamWriter hexwrite;

        /// <summary>
        /// Initialize the Hexinterpreter with no data
        /// </summary>
        public HexInterpreter(string filename)
        {
            FileName = filename;
        }

        public uint ReadHEX(VirtualPIC pic)
        {
            hexread = new StreamReader(FileName);
            pic.ProgMem = new byte[pic.ProgMemCapacity];
            for (int i = 0; i < pic.ProgMemCapacity; i++)
            { pic.ProgMem[i] = 0xFF; } //fill ProgMem with default erased value, 0xFF
            IsValid = true;
            HexLine line = new HexLine(hexread.ReadLine());
            while ((line.Type != 1) && !hexread.EndOfStream)
            {
                switch (line.Type)
                {
                    case 0: //normal data line
                        IntAddress = 0x0100 * line.Address[0] + line.Address[1];
                        if (AddressMSB == 0)//program memory
                        {
                            for (int i = 0; i < line.Length; i++)
                            { pic.ProgMem[i + IntAddress] = line.Data[i]; }
                        }
                        else if (AddressMSB == 0xF0) //EEPROM
                        {
                            if (pic.EEPROM == null)
                            {
                                pic.EEPROM = new byte[pic.EEPROMCapacity];
                                for (int i = 0; i < pic.EEPROMCapacity; i++)
                                { pic.EEPROM[i] = 0xFF; } //fill EEPROM with default erased value, 0xFF
                            }
                            for (int i = 0; i < line.Length; i++)
                            { pic.EEPROM[i + line.Address[1]] = line.Data[i]; }
                        }
                        else if (AddressMSB == 0x20) //ID locations
                        {
                            if (pic.IDlocs == null)
                                pic.IDlocs = new byte[8];
                            for (int i = 0; i < line.Length; i++)
                            { pic.IDlocs[i + line.Address[1]] = line.Data[i]; }
                        }
                        else if (AddressMSB == 0x30) //config
                        {
                            if (pic.ConfigWords == null)
                                pic.ConfigWords = new byte[pic.ConfigWordsNumBytes];
                            for (int i = 0; i < line.Length; i++)
                            { pic.ConfigWords[i + line.Address[1]] = line.Data[i]; }
                        }
                        break;
                    case 4: //extended linear address line
                        AddressMSB = line.Data[1];
                        break;
                    case 1: //EOF line
                    default:
                        break;
                }
                line = new HexLine(hexread.ReadLine());
                if (!line.IsValid())
                {
                    //MessageBox.Show("Error on line starting at: " + HexEncoding.ToString(line.Address));
                    IsValid = false;
                }
            }
            //Now remove empty space after end of program
            //First find last non-0xFF memory location starting at end
            int index,len;
            for (index = pic.ProgMemCapacity-1; index > 0; index--)
            {
                if (pic.ProgMem[index] != 0xFF)
                    break;
            }
            len = index-(index%16)+16;
            //create new array of required length, fill it with required program memory, replace ProgMem with it.
            byte[] temp = new byte[len];
            for (int i = 0; i < len; i++)
            { temp[i] = pic.ProgMem[i]; }
            pic.ProgMem = temp;
            hexread.Close();
            return 1;
        }

        public void WriteHEX(VirtualPIC pic)
        {
            hexwrite = new StreamWriter(FileName);
            hexwrite.WriteLine(":020000040000FA");//program memory
            ChunkWriteArray(ref pic.ProgMem, ref hexwrite);

            if (pic.EEPROM != null)
            {
                hexwrite.WriteLine(":0200000400F00A"); //EEPROM
                ChunkWriteArray(ref pic.EEPROM, ref hexwrite);
            }
            if (pic.IDlocs != null)
            {
                hexwrite.WriteLine(":020000040020DA"); //ID locations
                ChunkWriteArray(ref pic.IDlocs, ref hexwrite);
            }
            if (pic.ConfigWords != null)
            {
                hexwrite.WriteLine(":020000040030CA"); //Config Words
                ChunkWriteArray(ref pic.ConfigWords, ref hexwrite);
            }
            hexwrite.WriteLine(":00000001FF");//EOF line
            hexwrite.Close();
        }

        private void ChunkWriteArray(ref byte[] array, ref StreamWriter outfile)
        {
            int Len = 16;
            HexLine newline;

            for (int i = 0; i < array.Length; i += Len)
            {
                if ((array.Length - i) < Len)
                    Len = array.Length - i;
                newline = new HexLine(ref array, i, Len);
                if (!newline.IsEmpty())
                    outfile.WriteLine(newline.ToString());
            }
        }
    }

    class HexLine
    {
        private int Discarded;
        public byte Length, Type, Checksum;
        public byte[] Address = new byte[]{0,0};
        private byte[] ByteLine = new byte[] { };
        public byte[] Data;

        public HexLine(ref byte[] MemArray, int StartIndex, int DataLen)
        {
            Length = (byte)DataLen;
            Type = 0;
            Data = new byte[DataLen];
            Address[0] = (byte)((StartIndex / 0x100) % 0x100);
            Address[1] = (byte)(StartIndex % 0x100);
            Checksum = (byte)(DataLen + Address[0] + Address[1]);
            for (int i = 0; i < DataLen; i++)
            {
                Data[i] = MemArray[i + StartIndex];
                Checksum = (byte)(Checksum + Data[i]);
            }
            Checksum = (byte)(0x100 - Checksum);
        }

        public HexLine(string line)
        {
            ByteLine = HexEncoding.GetBytes(line.Substring(1,line.Length-1),out Discarded);
            Length = ByteLine[0];
            Address[0] = ByteLine[1];
            Address[1] = ByteLine[2];
            Type = ByteLine[3];
            Data = new byte[Length];
            for (int i = 0; i < Length; i++)
            {
                Data[i] = ByteLine[4 + i];
            }
            Checksum = ByteLine[ByteLine.Length-1];
        }

        public override string ToString()
        {
            return ":"+HexEncoding.ToString(new byte[]{(byte)Data.Length})+HexEncoding.ToString(Address)+"00"+HexEncoding.ToString(Data)+HexEncoding.ToString(new byte[]{Checksum});
        }
        ///<summary>
        ///returns true if hex file line is considered valid based on checksum
        ///</summary>
        public bool IsValid()
        {
            int temp;
            temp = Length + Address[0] + Address[1] + Type;
            for (int i = 0; i < Length; i++)
            {
                temp += Data[i];
            }
            temp = 0x100 - (temp % 0x100);
            if ((byte)temp != Checksum)
            {
                MessageBox.Show("Line: " + HexEncoding.ToString(Address) + " Checksum read: " + HexEncoding.ToString(new byte[] { Checksum }) + " Checksum calculated: " + HexEncoding.ToString(new byte[] { (byte)temp }));
            }
            return ((byte)temp == Checksum);
        }
        /// <summary>
        /// Returns true if the line contains _only_ 0xFF for all values
        /// </summary>
        /// <returns></returns>
        public bool IsEmpty()
        {
            for (int i = 0; i < Data.Length; i++)
            {
                if (Data[i] != 0xFF)
                    return false;
            }
            return true;
        }
    }

    //Thanks to Neilck on Codeproject for the HexEncoding class
    //http://www.codeproject.com/csharp/hexencoding.asp
    public class HexEncoding
    {
        public HexEncoding()
        {
            //
            // TODO: Add constructor logic here
            //
        }
        public static int GetByteCount(string hexString)
        {
            int numHexChars = 0;
            char c;
            // remove all none A-F, 0-9, characters
            for (int i = 0; i < hexString.Length; i++)
            {
                c = hexString[i];
                if (IsHexDigit(c))
                    numHexChars++;
            }
            // if odd number of characters, discard last character
            if (numHexChars % 2 != 0)
            {
                numHexChars--;
            }
            return numHexChars / 2; // 2 characters per byte
        }
        /// <summary>
        /// Creates a byte array from the hexadecimal string. Each two characters are combined
        /// to create one byte. First two hexadecimal characters become first byte in returned array.
        /// Non-hexadecimal characters are ignored. 
        /// </summary>
        /// <param name="hexString">string to convert to byte array</param>
        /// <param name="discarded">number of characters in string ignored</param>
        /// <returns>byte array, in the same left-to-right order as the hexString</returns>
        public static byte[] GetBytes(string hexString, out int discarded)
        {
            discarded = 0;
            string newString = "";
            char c;
            // remove all none A-F, 0-9, characters
            for (int i = 0; i < hexString.Length; i++)
            {
                c = hexString[i];
                if (IsHexDigit(c))
                    newString += c;
                else
                    discarded++;
            }
            // if odd number of characters, discard last character
            if (newString.Length % 2 != 0)
            {
                discarded++;
                newString = newString.Substring(0, newString.Length - 1);
            }

            int byteLength = newString.Length / 2;
            byte[] bytes = new byte[byteLength];
            string hex;
            int j = 0;
            for (int i = 0; i < bytes.Length; i++)
            {
                hex = new String(new Char[] { newString[j], newString[j + 1] });
                bytes[i] = HexToByte(hex);
                j = j + 2;
            }
            return bytes;
        }
        public static string ToString(byte[] bytes)
        {
            string hexString = "";
            for (int i = 0; i < bytes.Length; i++)
            {
                hexString += bytes[i].ToString("X2");
            }
            return hexString;
        }
        public static string ToString(byte[] bytes, bool spaceseparators)
        {
            string hexString = "";
            for (int i = 0; i < bytes.Length; i++)
            {
                hexString += bytes[i].ToString("X2")+" ";
            }
            return hexString;
        }
        /// <summary>
        /// Determines if given string is in proper hexadecimal string format
        /// </summary>
        /// <param name="hexString"></param>
        /// <returns></returns>
        public static bool InHexFormat(string hexString)
        {
            bool hexFormat = true;

            foreach (char digit in hexString)
            {
                if (!IsHexDigit(digit))
                {
                    hexFormat = false;
                    break;
                }
            }
            return hexFormat;
        }

        /// <summary>
        /// Returns true is c is a hexadecimal digit (A-F, a-f, 0-9)
        /// </summary>
        /// <param name="c">Character to test</param>
        /// <returns>true if hex digit, false if not</returns>
        public static bool IsHexDigit(Char c)
        {
            int numChar;
            int numA = Convert.ToInt32('A');
            int num1 = Convert.ToInt32('0');
            c = Char.ToUpper(c);
            numChar = Convert.ToInt32(c);
            if (numChar >= numA && numChar < (numA + 6))
                return true;
            if (numChar >= num1 && numChar < (num1 + 10))
                return true;
            return false;
        }
        /// <summary>
        /// Converts 1 or 2 character string into equivalant byte value
        /// </summary>
        /// <param name="hex">1 or 2 character string</param>
        /// <returns>byte</returns>
        private static byte HexToByte(string hex)
        {
            if (hex.Length > 2 || hex.Length <= 0)
                throw new ArgumentException("hex must be 1 or 2 characters in length");
            byte newByte = byte.Parse(hex, System.Globalization.NumberStyles.HexNumber);
            return newByte;
        }


    }
}
