/*
 * This source code has been downloaded from
 * http://eegeek.net/content/view/27/32/1/4/ on 15.11.2007
 * where it is/was made available under the
 * Creative Commons Attribution-Noncommercial-Share Alike 2.5 License.
 */

using System;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using USBBL;
using HEXReader;

using PVOID = System.IntPtr;
using DWORD = System.UInt32;

namespace usb_api
{
    unsafe public class Bootloader_Interface : usb_interface
    {
        protected override void OpenPipes()
        {
            DWORD selection = 0; // Selects the device to connect to
            myOutPipe = _MPUSBOpen(selection, vid_pid_boot, out_pipe, 0, 0); //using boot VID_PID
            myInPipe = _MPUSBOpen(selection, vid_pid_boot, in_pipe, 1, 0);
            prvPipesOpen = true;
        }

        public enum BootCommands : byte
        {
            READ_VERSION = 0x00,
            READ_FLASH = 0x01,
            WRITE_FLASH = 0x02,
            ERASE_FLASH = 0x03,
            READ_EEDATA = 0x04,
            WRITE_EEDATA = 0x05,
            READ_CONFIG = 0x06,
            WRITE_CONFIG = 0x07,
            UPDATE_LED = 0x32,
            RESET = 0xFF
        }

        /// <summary>
        /// Simple method for performing bootloader-specific tasks with the PIC.
        /// </summary>
        /// <param name="CMD">Command byte</param>
        /// <param name="length">Number of bytes to be read (READ operations only)</param>
        /// <param name="address">PIC memory address at which to perform the operation, if applicable</param>
        /// <param name="data">Data to be written (WRITE operations only)</param>
        /// <param name="dataout">Location for received data (if applicable)</param>
        /// <returns>1=Success, 2=Incorrect RX Length, 3=Read Failed</returns>
        public uint EasyBootCommand(BootCommands CMD, byte length, byte[] address, byte[] data, out byte[] dataout)
        {
            byte[] senddata;
            uint rval;

            switch (CMD)
            {
                case BootCommands.WRITE_FLASH:
                case BootCommands.WRITE_EEDATA:
                    senddata = new byte[4 + data.Length];
                    senddata[0] = (byte)data.Length;
                    senddata[1] = address[2]; //LSB first
                    senddata[2] = address[1];
                    senddata[3] = address[0];
                    //load all data bytes into buffer
                    for (int i = 0; i < data.Length; i++)
                    {
                        senddata[i + 4] = data[i];
                    }
                    rval = EasyCommand((byte)CMD, 1, senddata, out dataout);
                    if (rval != 1)
                        return rval;
                    break;
                case BootCommands.READ_EEDATA:
                case BootCommands.READ_CONFIG:
                case BootCommands.READ_FLASH:
                    senddata = new byte[4];
                    senddata[0] = (byte)length;
                    senddata[1] = address[2]; //LSB first
                    senddata[2] = address[1];
                    senddata[3] = address[0];
                    byte[] temp;
                    rval = EasyCommand((byte)CMD, 5 + (int)length, senddata, out temp);
                    dataout = new byte[length];
                    if (rval != 1)
                        return rval;
                    for (int i = 0; i < length; i++)
                        dataout[i] = temp[i + 5];
                    return 1;
                case BootCommands.ERASE_FLASH:
                    senddata = new byte[4];
                    senddata[0] = 1;
                    senddata[1] = address[2]; //LSB first
                    senddata[2] = address[1];
                    senddata[3] = address[0];
                    rval = EasyCommand((byte)CMD, 1, senddata, out dataout);
                    if (rval != 1)
                        MessageBox.Show("Error Code: " + rval.ToString());
                    return rval;
                case BootCommands.RESET:
                    senddata = new byte[] { };
                    rval = EasyCommand((byte)CMD, 0, senddata, out dataout);
                    return rval;
                default:
                    dataout = null;
                    return 0;
            }
            return 1;
        }

        private uint WriteFlashPacket(byte[] address, byte[] data)
        {
            uint rval;
            byte[] dataout;
            rval = EasyBootCommand(BootCommands.WRITE_FLASH, (byte)data.Length, address, data, out dataout);
            if (rval != 1)
                return rval;
            rval = EasyBootCommand(BootCommands.READ_FLASH, (byte)data.Length, address, null, out dataout);
            if (rval != 1)
                return rval;
            if (!Compare(dataout, data))
                return 5;
            return 1; //Write succeeded
        }

        private bool Compare(byte[] array1, byte[] array2)
        {
            if ((array1 == null) || (array2 == null))
                return false;
            if (array1.Length != array2.Length)
                return false;
            for (int i = 0; i < array1.Length; i++)
            {
                if (array1[i] != array2[i])
                    return false;
            }
            return true;
        }

        public uint Write(ref VirtualPIC vpic)
        {
            byte[] address = new byte[3];
            byte[] data = new byte[16];
            byte[] dataout;
            uint rval = this.Erase(vpic);
            //Write program memory
            for (int i = 0x0800; i < vpic.ProgMem.Length - 15; i += 16)
            {
                address[0] = 0;
                address[1] = (byte)((i / 0x100) % 0x100);
                address[2] = (byte)(i % 0x100);
                for (int j = 0; j < 16; j++)
                { data[j] = vpic.ProgMem[i + j]; }
                rval = WriteFlashPacket(address, data);
                if (rval != 1)
                    return 2;
            }
            if (vpic.EEPROM != null)
            {
                for (int i = 0; i < vpic.EEPROM.Length - 15; i += 16)
                {
                    address[0] = 0;
                    address[1] = (byte)((i / 0x100) % 0x100);
                    address[2] = (byte)(i % 0x100);
                    for (int j = 0; j < 16; j++)
                    { data[j] = vpic.EEPROM[i + j]; }
                    rval = EasyBootCommand(BootCommands.WRITE_EEDATA, 0, address, data, out dataout);
                    if (rval != 1)
                        return 3;
                    rval = EasyBootCommand(BootCommands.READ_EEDATA, (byte)data.Length, address, null, out dataout);
                    if (rval != 1)
                        return 3;
                    if (!Compare(dataout, data))
                        return 3;
                }
            }
            if (vpic.IDlocs != null)
            {
                address[0] = 0x20;
                address[1] = 0;
                address[2] = 0;

                //load 16-byte buffer for write
                byte[] temp = new byte[16];
                for (int i = 0; i < 16; i++)
                {
                    if (i < vpic.IDlocs.Length)
                        temp[i] = vpic.IDlocs[i];
                    else
                        temp[i] = 0xFF;
                }
                EasyBootCommand(BootCommands.ERASE_FLASH, 0, address, null, out dataout);
                EasyBootCommand(BootCommands.WRITE_FLASH, 0, address, temp, out dataout);
                EasyBootCommand(BootCommands.READ_FLASH, 16, address, null, out dataout);
                if (!Compare(dataout, temp))
                    return 4;
            }
            if (vpic.ConfigWords != null)
            {
                address[0] = 0x30;
                address[1] = 0;
                address[2] = 0;
                rval = EasyBootCommand(BootCommands.WRITE_CONFIG, 0, address, vpic.ConfigWords, out dataout);
                if (!Compare(dataout, vpic.ConfigWords))
                    return 5;
            }
            return 1;
        }

        public uint Read(ref VirtualPIC vpic)
        {
            vpic.ProgMem = new byte[vpic.ProgMemCapacity];
            int len = 59;
            uint rval;
            byte[] buf = new byte[len];
            for (int i = 0; i < vpic.ProgMem.Length - 1; i += len)
            {
                if (i + len > vpic.ProgMemCapacity - 1)//read appropriate number of bytes if <59 left to end of memory space
                    len = vpic.ProgMemCapacity - i;
                rval = EasyBootCommand(BootCommands.READ_FLASH, (byte)len, new byte[] { 0, (byte)((i / 0x100) % 0x100), (byte)(i % 0x100) }, null, out buf);
                if (rval != 1)
                    return rval;
                for (int j = 0; j < len; j++)
                    vpic.ProgMem[i + j] = buf[j];
            }
            vpic.EEPROM = new byte[vpic.EEPROMCapacity];
            len = 59;
            for (int i = 0; i < vpic.EEPROM.Length - 1; i += len)
            {
                if (i + len > vpic.EEPROM.Length)//read appropriate number of bytes if <59 left to end of memory space
                    len = vpic.EEPROM.Length - i;
                rval = EasyBootCommand(BootCommands.READ_EEDATA, (byte)len, new byte[] { 0, 0, (byte)i }, null, out buf);
                if (rval != 1)
                    return rval;
                for (int j = 0; j < len; j++)
                    vpic.EEPROM[i + j] = buf[j];
            }
            vpic.IDlocs = new byte[8];
            rval = EasyBootCommand(BootCommands.READ_CONFIG, 8, new byte[] { 0x20, 0, 0 }, null, out buf);
            if (rval != 1)
                return rval;
            for (int j = 0; j < 8; j++)
                vpic.IDlocs[j] = buf[j];
            vpic.ConfigWords = new byte[14];
            rval = EasyBootCommand(BootCommands.READ_CONFIG, 14, new byte[] { 0x30, 0, 0 }, null, out buf);
            if (rval != 1)
                return rval;
            for (int j = 0; j < 14; j++)
                vpic.ConfigWords[j] = buf[j];
            return 1;
        }

        public uint Execute()
        {
            byte[] asdf = new byte[] { };
            uint rval = EasyBootCommand(BootCommands.RESET, 1, asdf, asdf, out asdf);
            return rval;
        }

        public uint Erase(int ProgMemCapacity, int EEPROMCapacity)
        {
            uint rval;
            byte[] address = new byte[3] { 0, 0, 0 };
            byte[] dataout;
            for (int i = 0x0800; i < ProgMemCapacity; i += 64)
            {
                address[1] = (byte)((i / 0x100) % 0x100);
                address[2] = (byte)(i % 0x100);
                rval = EasyBootCommand(BootCommands.ERASE_FLASH, 1, address, null, out dataout);
                if (rval != 1)
                    return rval;
            }
            //"erase" EEPROM by overwriting with all 0xFF
            byte[] data = new byte[16] { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
            for (int i = 0; i < EEPROMCapacity; i += 16)
            {
                address[1] = (byte)((i / 0x100) % 0x100);
                address[2] = (byte)(i % 0x100);
                rval = EasyBootCommand(BootCommands.WRITE_EEDATA, (byte)data.Length, address, data, out dataout);
                if (rval != 1)
                    return rval;
            }
            //erase ID locations
            EasyBootCommand(BootCommands.ERASE_FLASH, 1, new byte[] { 0x20, 0, 0 }, null, out dataout);
            return 1;
        }

        public uint Erase(VirtualPIC vpic)
        {
            return Erase(vpic.ProgMemCapacity, vpic.EEPROMCapacity);
        }

    }
}
