﻿/*
 * This source code has been originally downloaded from
 * www.steffenschuette.de where it is/was made available under the license:
 * Creative Commons Attribution-Noncommercial-Share Alike 2.0 Germany
 * 
 * If you want to modify and redistribute your code please add your name 
 * and changes here:
 * 
 * Date         Author            Change
 * 14.02.2008 | Steffen Schütte | Initial version
 */

using System;
using System.Collections.Generic;
using System.Text;
using StarUML;
using StarUML_PICAddIn.codeGen.Objects;
using StarUML_PICAddIn.codeGen.Activities;

namespace StarUML_PICAddIn.codeGen
{
    internal abstract class VisualElement
    {
        private IUMLModelElement modelElement;

        public IUMLModelElement ModelElement
        {
            get { return modelElement; }
        }

        private IList<VisualElement> predecessors;

        internal IList<VisualElement> Predecessors
        {
            get { return predecessors; }
        }

        private IList<DirectedTransition> successors;

        internal IList<DirectedTransition> Successors
        {
            get { return successors; }
        }


        public VisualElement(IUMLModelElement modelElement)
        {
            this.modelElement = modelElement;
            successors = new List<DirectedTransition>();
            predecessors = new List<VisualElement>();
        }

        public abstract bool IsObject();

        public virtual String GenerateInitCode()
        {
            return "";//Default is no init code
        }

        public IList<VisualElement> ConsumedObjects
        {
            get
            {
                IList<VisualElement> returnList = new List<VisualElement>();

                foreach(VisualElement e in predecessors)
                {
                    if (e.IsObject())
                    {
                        returnList.Add(e);
                    }
                }

                return returnList;
            }
        }

        public IList<VisualElement> ProducedObjects
        {
            get
            {

                IList<VisualElement> returnList = new List<VisualElement>();

                foreach (DirectedTransition t in successors)
                {
                    if (t.Successor.IsObject())
                    {
                        returnList.Add(t.Successor);
                    }
                }

                return returnList;
            }
        }

        public IList<VisualElement> SuccessorActivities
        {
            get
            {
                IList<VisualElement> returnList = new List<VisualElement>();

                foreach (DirectedTransition t in successors)
                {
                    if (!t.Successor.IsObject())
                    {
                        returnList.Add(t.Successor);
                    }
                }

                return returnList;
            }
        }


        protected bool CheckConsumed(uint minAllowedObjects, uint maxAllowedObjects)
        {
            if (ConsumedObjects.Count < minAllowedObjects || ConsumedObjects.Count > maxAllowedObjects)
            {
                throw new Exception(this.ModelElement.Name + ": Wrong number of consumed objects. Must be within [" + minAllowedObjects + ".." + maxAllowedObjects + "]");
            }
            return true;
        }


        protected bool CheckProduced(uint minAllowedObjects, uint maxAllowedObjects)
        {
            if (ProducedObjects.Count < minAllowedObjects || ProducedObjects.Count > maxAllowedObjects)
            {
                CreationProtocol.GetInstance().AddMessage(MessageType.Error, this.ModelElement.Name + ": Wrong number of produced objects. Must be within [" + minAllowedObjects + ".." + maxAllowedObjects +"]" );
            }
            return true;
        }

        protected bool CheckSuccessorActivities(uint minAllowedObjects, uint maxAllowedObjects)
        {
            if (SuccessorActivities.Count < minAllowedObjects || SuccessorActivities.Count > maxAllowedObjects)
            {
                CreationProtocol.GetInstance().AddMessage(MessageType.Error, this.ModelElement.Name + ": Wrong number of succeding activities. Must be within [" + minAllowedObjects + ".." + maxAllowedObjects + "]");
            }
            return true;
        }



        /// <summary>
        /// Creates the corresponding visual element for the given
        /// UMLModelElement.
        /// 
        /// FactoryMethod
        /// </summary>
        /// <param name="modelElement">The model element.</param>
        /// <returns></returns>
        public static VisualElement CreateVisualElementFor(IUMLModelElement modelElement)
        {
            //TODO:
            //This could be done more elegant using reflection and the stereotype as class name

            if(modelElement.StereotypeName.StartsWith("BitIn"))
            {
                return new BitIn(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("BitOut"))
            {
                return new BitOut(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("Flowstart"))
            {
                return new FlowStart(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("Assignment"))
            {
                return new Assignment(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("PWMOut"))
            {
                return new PWMOut(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("Constant"))
            {
                return new Constant(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("Variable"))
            {
                return new Variable(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("Addition"))
            {
                return new Addition(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("IsZero"))
            {
                return new EqualToZero(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("GreaterZero"))
            {
                return new GreaterZero(modelElement);
            }
            else if (modelElement.StereotypeName.StartsWith("Delay"))
            {
                return new Delay(modelElement);
            }

            return null;
        }


        public override string ToString()
        {
            return modelElement.Name;
        }
    }
}
