﻿/*
 * This source code has been originally downloaded from
 * www.steffenschuette.de where it is/was made available under the license:
 * Creative Commons Attribution-Noncommercial-Share Alike 2.0 Germany
 * 
 * If you want to modify and redistribute your code please add your name 
 * and changes here:
 * 
 * Date         Author            Change
 * 14.02.2008 | Steffen Schütte | Initial version
 */

using System;
using System.Collections.Generic;
using System.Text;
using StarUML;
using StarUML_PICAddIn.codeGen;
using StarUML_PICAddIn.codeGen.Activities;

namespace StarUML_PICAddIn
{
    internal class VisualCodeGenerator
    {
        private IStarUMLApplication starUML;

        /// <summary>
        /// Mapping from stereotypes to visual elements
        /// </summary>
        private IDictionary<string, IList<VisualElement>> stereotypeDict;

        /// <summary>
        /// Mapping from model elements to visual elements.
        /// </summary>
        private IDictionary<IUMLModelElement, VisualElement> visualElementDict;

        private IList<string> includes;

        private IDictionary<string, string> globalVarsToCreate;

        public VisualCodeGenerator(IStarUMLApplication starUML)
        {
            this.starUML = starUML;
            stereotypeDict = new Dictionary<string, IList<VisualElement>>();
            visualElementDict = new Dictionary<IUMLModelElement, VisualElement>();
            globalVarsToCreate = new Dictionary<string, string>();
            includes = new List<string>();

            //Scan the application for elements
            GetModelElements(0, starUML.GetProject());
        }

        private void GetModelElements(int level, IUMLNamespace namespc)
        {
            Console.WriteLine(level + ": " + namespc.Name);
       
            for (int i = 0; i < namespc.GetOwnedElementCount(); i++)
            {
                if (namespc.GetOwnedElementAt(i).IsKindOf("UMLNamespace"))
                {
                    GetModelElements(level + 1, (IUMLNamespace)namespc.GetOwnedElementAt(i));
                }
                else
                {
                    //AddElementToDict(namespc.GetOwnedElementAt(i));
                }
            }
            for (int i = 0; i < namespc.GetBehaviorCount(); i++)
            {
                ElaborateBehaviour(namespc.GetBehaviorAt(i));
            }

        }

        private void ElaborateBehaviour(IUMLStateMachine stateMachine)
        {
            for (int i = 0; i < stateMachine.GetTransitionCount(); i++)
            {
                IUMLTransition t = stateMachine.GetTransitionAt(i);
                AddTransitionToDict(t);
            } 
        }

        /// <summary>
        /// Adds the transition to the dictionary.
        /// The corresponding source and target objects are created also
        /// and connected.
        /// </summary>
        /// <param name="transition">The transition.</param>
        private void AddTransitionToDict(IUMLTransition transition)
        {
            string guardCondition = transition.GuardCondition;

            //Get or create visual source element
            VisualElement visualSource;
            IUMLModelElement modelSource = transition.Source;
            if (!visualElementDict.TryGetValue(modelSource, out visualSource))
            {
                visualSource = VisualElement.CreateVisualElementFor(modelSource);
                visualElementDict.Add(modelSource, visualSource);

                //Add to stereotype dict as well
                AddToStereotypeDict(visualSource);
            }

            //Get or create visual target element
            VisualElement visualTarget;
            IUMLModelElement modelTarget = transition.Target;
            if (!visualElementDict.TryGetValue(modelTarget, out visualTarget))
            {
                visualTarget = VisualElement.CreateVisualElementFor(modelTarget);
                visualElementDict.Add(modelTarget, visualTarget);

                //Add to stereotype dict as well
                AddToStereotypeDict(visualTarget);
            }

            //Now add the elements to each others successors/predecessors
            visualSource.Successors.Add(new DirectedTransition(visualTarget, guardCondition));
            visualTarget.Predecessors.Add(visualSource);

        }

        private void AddToStereotypeDict(VisualElement visualElement)
        {
            IUMLModelElement modelElement = visualElement.ModelElement;
  
            if (modelElement.StereotypeName != null && modelElement.StereotypeName.Length > 0)
            {
                string stereotype = modelElement.StereotypeName.Substring(0, modelElement.StereotypeName.IndexOf(" ")) ;

                IList<VisualElement> elementList;
                if (stereotypeDict.TryGetValue(stereotype, out elementList))
                {
                    //Add element if not already existing
                    if (!elementList.Contains(visualElement))
                    {
                        elementList.Add(visualElement);
                    }
                }
                else
                {
                    //Create a list and add it to the dictionary
                    elementList = new List<VisualElement>();
                    elementList.Add(visualElement);
                    stereotypeDict.Add(stereotype, elementList);
                }
            }
        }
         

        /// <summary>
        /// Generates the code for the specified starUML application.
        /// </summary>
        /// <returns></returns>
        internal string GenerateLoop()
        {
            StringBuilder code = new StringBuilder();


            if (!stereotypeDict.ContainsKey("Flowstart"))
            {
                //There must be at least one initial activity
                CreationProtocol.GetInstance().AddMessage(MessageType.Error, "There must be a 'Flowstart' activity connected to another activity.");
            }
            else
            {
                int numOfInitialStates = stereotypeDict["Flowstart"].Count;
                if (numOfInitialStates == 1)
                {
                    //Elaborate the initial activity
                    try
                    {
                        ((IActivity)stereotypeDict["Flowstart"][0]).GenerateCode(code);
                    }
                    catch (Exception ex)
                    {
                        CreationProtocol.GetInstance().AddMessage(MessageType.Error, ex.Message);
                    }
                }
                else
                {
                    //Too many initial activities
                    CreationProtocol.GetInstance().AddMessage(MessageType.Error, "There must only be one 'Flowstart' activity.");
                }
            }
            return code.ToString();
        }

        /// <summary>
        /// Generates the init code
        /// 
        /// TODO: This could be moved to the different subclasses
        /// Problem: Take care of confilcts!
        /// </summary>
        /// <returns></returns>
        internal string GenerateInit()
        {
            StringBuilder code = new StringBuilder();

            //This list takes care of double ressources as there can be 
            //several UML elements pointing to the same PIC ressource
            List<string> initializedResources = new List<string>();

            //Generate init code for BitIn elements
            if(stereotypeDict.ContainsKey("BitIn"))
            {
                code.AppendLine("\t//BitIn initialization...");
                initializedResources.Clear();           
                foreach(VisualElement visualElement in stereotypeDict["BitIn"])
                {
                    IUMLModelElement bitIn = visualElement.ModelElement;

                    string port = bitIn.GetTaggedValueAsString("VisualPIC(18F4550)", "BitInOutTags", "Port");
                    string pin = bitIn.GetTaggedValueAsString("VisualPIC(18F4550)", "BitInOutTags", "Pin");

                    code.AppendLine("\tTRIS" + port + "bits.TRIS" + port + pin + " = 1;");

                    //TODO: Check for double usage using initializedRessources list
                }
            }

            //Generate init code for BitOut elements
            if (stereotypeDict.ContainsKey("BitOut"))
            {
                code.AppendLine("\t//BitOut initialization...");
                initializedResources.Clear();
                foreach (VisualElement visualElement in stereotypeDict["BitOut"])
                {
                    IUMLModelElement bitOut = visualElement.ModelElement;

                    string port = bitOut.GetTaggedValueAsString("VisualPIC(18F4550)", "BitInOutTags", "Port");
                    string pin = bitOut.GetTaggedValueAsString("VisualPIC(18F4550)", "BitInOutTags", "Pin");

                    code.AppendLine("\tTRIS" + port + "bits.TRIS" + port + pin + " = 0;");
                    code.AppendLine("\tPORT" + port + "bits.R" + port + pin + " = 0;");

                    //TODO: Check for double usage/in/out usage conflicts initializedRessources list
                }
            }

            //Generate init code for PWMOut elements
            if (stereotypeDict.ContainsKey("PWMOut"))
            {
                //Add include for C18 PWM-library
                includes.Add("<PWM.h>");
                includes.Add("<timers.h>");

                code.AppendLine("\t//PWMOut initialization...");
                initializedResources.Clear();
                foreach (VisualElement visualElement in stereotypeDict["PWMOut"])
                {
                    IUMLModelElement pwmOut = visualElement.ModelElement;

                    string initialDuty = pwmOut.GetTaggedValueAsString("VisualPIC(18F4550)", "PWMTags", "InitialDuty(0-1023)");

                    code.AppendLine("\tOpenTimer2(T0_8BIT");
                    code.AppendLine("\t& T0_SOURCE_INT");
                    code.AppendLine("\t& T0_PS_1_256");
                    code.AppendLine("\t& TIMER_INT_ON");
                    code.AppendLine("\t);");

                    code.AppendLine("\tOpenPWM1(0b11111111);");
                    code.AppendLine("\tSetDCPWM1(" + initialDuty + ");");

                }
            }

            if(stereotypeDict.ContainsKey("Delay"))
            {
                //Add include for C18 delay-library
                includes.Add("<delays.h>");
            }




            return code.ToString(); 
        }



        internal string GenerateGlobalVars()
        {
            StringBuilder code = new StringBuilder();

            //Generate init code for BitIn elements
            if(stereotypeDict.ContainsKey("Variable"))
            {
                foreach (VisualElement visualElement in stereotypeDict["Variable"])
                {
                    IUMLModelElement variable = visualElement.ModelElement;
                    //TODO: get type string port = bitIn.GetTaggedValueAsString("VisualPIC(18F4550)", "BitInOutTags", "Port");
                    code.AppendLine("int " + variable.Name.Replace(" ", "_") + ";" );
                }
            }

            //Add other required global vars
            foreach(String key in globalVarsToCreate.Keys)
            {
                code.Append(globalVarsToCreate[key]);
                code.Append(" ");
                code.Append(key);
                code.AppendLine(";");
            }
            return code.ToString();
        }



        /// <summary>
        /// Generates the includes.
        /// </summary>
        /// <returns></returns>
        internal string GenerateIncludes()
        {
            StringBuilder includeCode = new StringBuilder();
            foreach(string include in includes)
            {
                includeCode.Append("#include ");
                includeCode.AppendLine(include);
            }

            return includeCode.ToString();
        }
    }
}
