using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using usb_api;

namespace UltrasonicUSBRadar
{
    public partial class Form1 : Form
    {
        private usb_interface usbi;

        private PICAsyncManager pam;

        private TargetInfo targetInfo = new TargetInfo();

        //private uint antennaPosition = 0;

        private MissileController missileControl;

        

        public Form1()
        {
            InitializeComponent();

            pam = new PICAsyncManager();

            usbi = new usb_interface();

            missileControl = new MissileController(targetInfo);

            this.Controls.Add(pam);
            pam.EventNewUSBIntData += new NewUSBIntDataEventHandler(pam_EventNewUSBIntData);

            radarScreen1.TargetInfo = targetInfo;
            radarScreen1.MissileController = missileControl;
                     
         
            missileControl.LauncherMessage_Event += new MissileController.LauncherMessage_EventHandler(missileControl_LauncherMessage_Event);
            missileControl.LauncherDisarmed_Event += new MissileController.LauncherDisarmed_EventHandler(missileControl_LauncherDisarmed_Event);

            CheckLauncherConnected();
        }

        private void CheckLauncherConnected()
        {
                checkBox1.Checked = missileControl.IsLauncherConnected();
        }

        void missileControl_LauncherDisarmed_Event()
        {
            DisarmLauncher();
        }

        private void DisarmLauncher()
        {
                buttonArmMissile.Text = "Arm";
                labelMissileState.BackColor = labelMissileState.BackColor = Color.Transparent;
                labelMissileState.Text = "Disarmed";
                missileControl.Dismantle();
                targetInfo.ActiveTarget = new KeyValuePair<uint, uint>(uint.MaxValue, uint.MaxValue);

                buttonDown.Enabled = true;
                buttonUp.Enabled = true;
                buttonLeft.Enabled = true;
                buttonRight.Enabled = true;
                buttonFire.Enabled = true;

                missileControl.LauncherStop();
        }

        void missileControl_LauncherMessage_Event(string msg)
        {
            toolStripStatusLabel1.Text = msg;
        }

      


        private void buttonStartStop_Click(object sender, EventArgs e)
        {
            setStepsButton.PerformClick();

            if (!pam.IsRunning)
            {
                pam.Start();
                toolStripStatusLabel1.Text = "Interrupt monitor thread started...";
                buttonStartStop.Text = "Stop";
            }
            else
            {
                pam.Stop();
                toolStripStatusLabel1.Text = "Interrupt monitor thread stopped...";
                buttonStartStop.Text = "Start";
            }
        }

        /// <summary>
        /// New data has been received from the pic.
        /// </summary>
        /// <param name="e">The <see cref="usb_api.NewUSBIntDataEventArgs"/> instance containing the event data.</param>
        void pam_EventNewUSBIntData(NewUSBIntDataEventArgs e)
        {
            try
            {
                for (int i = 0; i <= 3; i += 3)
                {
                    float stepsPerDegree = float.Parse(degPerStepsTextBox.Text);

                    //Get antenna position
                    targetInfo.antennaPosition = (uint)(e.newdata[i] * 7.5);

                    //Get 8 byte values
                    byte dataL = e.newdata[i + 1];
                    byte dataH = e.newdata[i + 2];

                    //Create 16bit value
                    uint data = dataL + (uint)(256 * dataH);

                    //21,248 uS = 1 tick
                    //Now data is the number of elapsed uS
                    data = (uint)((double)data * 21.248);

                    //Now get the distance in cm (see SRF04 datasheet)
                    data = data / 58;

                    if(data <= 300)
                    {
                        //Update the data for the given antenna position or create it
                        if (!targetInfo.Targets.ContainsKey(targetInfo.antennaPosition))
                        {
                            targetInfo.Targets.Add(targetInfo.antennaPosition, data);
                        }
                        else
                        {
                            targetInfo.Targets[targetInfo.antennaPosition] = data;
                        }
                    }
                    else
                    {
                        targetInfo.Targets.Remove(targetInfo.antennaPosition);
                    }

                }
                //targets.antennaPosition = antennaPosition;

                //Redraw the radar screen
                radarScreen1.Refresh();

                //Run target detection if missile is armed
                if(missileControl.LauncherArmed)
                {
                    DetectTargets();
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                buttonStartStop.PerformClick();
            }
        }


        /// <summary>
        /// Sets the number of steps to do into the pic
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void setStepsButton_Click(object sender, EventArgs e)
        {
            byte numberOfSteps = (byte)numOfStepsNumericUpDown.Value;

            byte[] temp = new byte[1];
            byte[] temp2;
            usbi.EasyCommand(0x33, 1, new byte[] { numberOfSteps }, out temp2);
        }


        private void buttonArmMissile_Click(object sender, EventArgs e)
        {
            try
	        {	        
		        missileControl.DegrePerSecond = float.Parse(textBoxLauncherSpeed.Text);
	
               
                if (!missileControl.LauncherArmed)
                {
                    missileControl.Arm();
                    buttonArmMissile.Text = "Dismantle";
                    labelMissileState.Text = "Armed!!!";
                    labelMissileState.BackColor = Color.Red;
                    buttonDown.Enabled = false;
                    buttonUp.Enabled = false;
                    buttonLeft.Enabled = false;
                    buttonRight.Enabled = false;
                    buttonFire.Enabled = false;

                    //Set configuration values
                    targetInfo.DetectionOffset = (uint)numericUpDownDetectionOffset.Value;
                    targetInfo.DetectionRange = (uint)numericUpDownDetectionRange.Value;
                    targetInfo.DetectionTolerance = (uint)numericUpDownDetectionTolerance.Value;
                    missileControl.DegrePerSecond = float.Parse(textBoxLauncherSpeed.Text);
                    
                }
                else
                {
                    DisarmLauncher();
                }

          

            }
            catch (Exception ex)
            {
                MessageBox.Show(this, ex.Message);
            }
        }

        private void DetectTargets()
        {
            uint distance = uint.MaxValue;
            uint degree = 0;
            bool match = false;
            foreach(KeyValuePair<uint, uint> kvp in targetInfo.Targets)
            {
                if(targetInfo.TargetingRange.ContainsKey(kvp.Key))
                {
                    if(kvp.Value < targetInfo.TargetingRange[kvp.Key])
                    {
                        //TODO check if this is within the detection range!
                        if (IsWithinDetectionRange(kvp.Key))
                        {
                            if (kvp.Value < distance)
                            {
                                match = true;
                                distance = kvp.Value;
                                degree = kvp.Key;
                            }
                        }
                    }
                }
            }

            if (match)
            {
                if (missileControl.MissilesAvailable > 0)
                {
                    missileControl.Launch(degree);
                    targetInfo.ActiveTarget = new KeyValuePair<uint, uint>(degree, distance);
                }
                else
                {
                    DisarmLauncher();
                }
            }
            else
            {
                missileControl.StopTheLaunch();
                targetInfo.ActiveTarget = new KeyValuePair<uint,uint>(uint.MaxValue, uint.MaxValue);
            }
            
        }

        void Form1_FormClosed(object sender, System.Windows.Forms.FormClosedEventArgs e)
        {
            //Stop targeting thread to exit the app
            missileControl.TargetingThread.Abort();
        }


        private bool IsWithinDetectionRange(uint p)
        {
            uint offset = targetInfo.DetectionOffset;
            uint range = targetInfo.DetectionRange;

            if(p >= offset && p <= (offset+range))
            {
                return true;
            }
            else if(offset + range > 360)
            {
                if(p <= (offset+range)%360)
                {
                    return true;
                }
            }
            return false;
        }



        private void buttonScanEnvironment_Click(object sender, EventArgs e)
        {
            targetInfo.CreateEnvironmentFormTargets();
            targetInfo.CreateDetectionRangeFromEnvironment();
            radarScreen1.Refresh();
        }

        private void numericUpDownDetectionOffset_ValueChanged(object sender, EventArgs e)
        {
            if(numericUpDownDetectionOffset.Value + numericUpDownDetectionRange.Value > 360)
            {
                numericUpDownDetectionRange.Value = 360 - numericUpDownDetectionOffset.Value;
                
            }
            targetInfo.DetectionOffset = (uint)numericUpDownDetectionOffset.Value;
                radarScreen1.Refresh();    
          
        }

        private void numericUpDownDetectionRange_ValueChanged(object sender, EventArgs e)
        {
           if (numericUpDownDetectionOffset.Value + numericUpDownDetectionRange.Value > 359)
            {
                numericUpDownDetectionOffset.Value = 360 - numericUpDownDetectionRange.Value;
                
            }
            targetInfo.DetectionRange = (uint)numericUpDownDetectionRange.Value;
            radarScreen1.Refresh();  
      
          
        }

        private void numericUpDownDetectionTolerance_ValueChanged(object sender, EventArgs e)
        {
            targetInfo.DetectionTolerance = (uint)numericUpDownDetectionTolerance.Value;
            targetInfo.CreateDetectionRangeFromEnvironment();
            radarScreen1.Refresh();
        }



        void buttonDown_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if(this.missileControl.LauncherConnected)
            {
                missileControl.LauncherDown();
            }
        }

        void buttonUp_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (this.missileControl.LauncherConnected)
            {
                missileControl.LauncherStop();
            }
        }

        void buttonUp_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (this.missileControl.LauncherConnected)
            {
                missileControl.LauncherUp();
            }
        }

        private void checkBox1_CheckedChanged(object sender, EventArgs e)
        {
            this.missileControl.LauncherConnected = checkBox1.Checked;
        }

        private void buttonFire_Click(object sender, EventArgs e)
        {
            this.missileControl.Fire();
        }

        void buttonRight_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (this.missileControl.LauncherConnected)
            {
                missileControl.LauncherRight();
            }
        }

        void buttonLeft_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (this.missileControl.LauncherConnected)
            {
                missileControl.LauncherLeft();
            }
        }
      
    }
}