/*********************************************************************
 *
 *                Microchip USB C18 Firmware Version 1.0
 *
 *********************************************************************
 * FileName:        user.c
 * Dependencies:    See INCLUDES section below
 * Processor:       PIC18
 * Compiler:        C18 2.30.01+
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the Company) for its PICmicro Microcontroller is intended and
 * supplied to you, the Companys customer, for use solely and
 * exclusively on Microchip PICmicro Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Rawin Rojvanit       11/19/04    Original.
 ********************************************************************/

/** I N C L U D E S **********************************************************/
#include <p18cxxx.h>
#include <delays.h>
//#include <usart.h>
#include "system\typedefs.h"

#include "system\usb\usb.h"

#include "io_cfg.h"             // I/O pin mapping
#include "user\user.h"

/** V A R I A B L E S ********************************************************/
#pragma udata

byte counter; //Number of bytes to trsnsmit in one chunk
//byte trf_state;
byte countval;
//byte lastcountval;

DATA_PACKET dataPacket;


//Stepper control
byte out1[4] = {1, 1, 0, 0};
byte out2[4] = {0, 0, 1, 1};
byte out3[4] = {1, 0, 0, 1};
byte out4[4] = {0, 1, 1, 0};
byte stepperPos = 0;

byte stepsDone;

byte stepsToDo;

//Radar control
byte numOfPings;

byte advanceMotor;

enum
{
    RADAR_SEND      = 0x00, //Send a ping
    RADAR_RECEIVE_1 = 0x01, //Receive the ping
    RADAR_RECEIVE_0 = 0x02, //Receive the end of the ping
    RADAR_STEP		= 0x03,  //Advance the stepper motor
    RADAR_TURNBACK	= 0x04  //Turn back the stepper motor to the initial position
}radarPhase;

/** P R I V A T E  P R O T O T Y P E S ***************************************/

void BlinkUSBStatus(void);
void ServiceRequests(void);


void StepForward(void);
void StepBackward(void);

/** D E C L A R A T I O N S **************************************************/
#pragma code
void UserInit(void)
{
    mInitAllLEDs();
 
	//Configure TIMER3 for stepper control
	T3CON = 0b00110001; //timer3 on, 1:2 prescaler
	PIE2bits.TMR3IE = 1;
	INTCONbits.PEIE = 1;
	countval = 0;

	//Configure TIMER0 for echo measurement
    //T0CON = 0b11000111;	
    
    T0CONbits.T0PS0 = 1;
    T0CONbits.T0PS1 = 1;
    T0CONbits.T0PS2 = 1;
    T0CONbits.PSA = 0;
    T0CONbits.T0SE = 0;
    T0CONbits.T0CS = 0;
    T0CONbits.T08BIT = 0;
	T0CONbits.TMR0ON = 0;


	//T1CON = 0b10111001;	
	
	ADCON1 = 0x0F;                 // Default all pins to digital
	
	//Stepper BitOut initialization...
	TRISAbits.TRISA0 = 0;
	PORTAbits.RA0 = 0;
	TRISAbits.TRISA1 = 0;
	PORTAbits.RA1 = 0;
	TRISAbits.TRISA2 = 0;
	PORTAbits.RA2 = 0;
	TRISAbits.TRISA3 = 0;
	PORTAbits.RA3 = 0;
	TRISAbits.TRISA4 = 0;
	PORTAbits.RA4 = 0;
	

	// Set pin RB3 to an output 
	// Connects to "Trigger In" on SRF04
	TRISBbits.TRISB3 = 0;
	PORTBbits.RB3 = 0;

	// Set pin RB7 to an input
	// Connects to "Echo Out" on SRF04
	TRISBbits.TRISB7 = 1;
	
	numOfPings = 0;
	radarPhase = RADAR_SEND;
	advanceMotor = 0;
	stepsDone = 0;
	
	stepsToDo = 10;
}//end UserInit


/******************************************************************************
 * Function:        void ProcessIO(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is a place holder for other user routines.
 *                  It is a mixture of both USB and non-USB tasks.
 *
 * Note:            None
 *****************************************************************************/
void ProcessIO(void)
{   
	byte index;
	byte distanceL;
	byte distanceH;

	INTCONbits.GIE = 1;
	// User Application USB tasks go here
	
	//Operate the radar if no data is pending
	if(!mUSBIntTxIsBusy() && numOfPings < 2)
	{
		switch(radarPhase)
		{
			case RADAR_SEND:
				mLED_5_On();
			    mLED_6_Off();
			    mLED_7_Off();
			    mLED_8_Off();
			    
				//Start SRF04 measurement
				LATBbits.LATB3 = 1; 	// output high voltage to pin
				Delay10TCYx( 13 ); 	// wait the desired pulse width (>10uS)
			    LATBbits.LATB3 = 0; 	// output 0 voltage to pin
			    
			    //Advance to next phase
			    radarPhase = RADAR_RECEIVE_1;

				break;    
				
			case RADAR_RECEIVE_1:
				mLED_5_Off();
			    mLED_6_On();
			    mLED_7_Off();
			    mLED_8_Off();
			    
				if(PORTBbits.RB7 == 1)
				{
					//Reset the timer
					TMR0H = 0; //Write high first as it is updated when tmr0L is written
					TMR0L = 0;
					T0CONbits.TMR0ON = 1;
					
					//Advance to next phase
				    radarPhase = RADAR_RECEIVE_0;					
				}			    
				break;    
		
			case RADAR_RECEIVE_0:
			    mLED_5_Off();
			    mLED_6_Off();
			    mLED_7_On();
			    mLED_8_Off();
			    if(PORTBbits.RB7 == 0)
			    {
				    //Read out the timer
				    T0CONbits.TMR0ON = 0;
				    distanceL = TMR0L;
   				    distanceH = TMR0H;
				    
				    //Set the data to send
				    index = numOfPings*3;
		    		dataPacket._byte[index] = stepsDone; //The current position
					dataPacket._byte[index + 1] = distanceL; //The distance
					dataPacket._byte[index + 2] = distanceH; //The distance
				    
				    numOfPings++;
				    //Advance to next phase
					radarPhase = RADAR_STEP;
					
					//Start the timer with 10ms to slow down the motor and let the SRF04 recover				    
					TMR3H = 0;
					TMR3L = 1;
					PIR2bits.TMR3IF = 0;
					advanceMotor = 0;
				}
				break;
				
			case RADAR_STEP:
			    mLED_5_Off();
			    mLED_6_Off();
			    mLED_7_Off();
			    mLED_8_On();
			    //Step the motor if the timer is over
			    if(advanceMotor == 1)
			    {
					//Advance the motor
					StepForward();
					TMR3H = 0;
					TMR3L = 1;
					PIR2bits.TMR3IF = 0;
					advanceMotor = 0;
					stepsDone++;
					
					if(stepsDone > stepsToDo)
					{
						radarPhase = RADAR_TURNBACK;
					}
					else
					{
					    //Advance to next phase
					    radarPhase = RADAR_SEND;						
					}
			    }
				break;   
			case RADAR_TURNBACK:
				mLED_5_Off();
			    mLED_6_Off();
			    mLED_7_Off();
			    mLED_8_Off();
			    //Step the motor if the timer is over
			    if(advanceMotor == 1)
			    {
					//Advance the motor
					StepBackward();
					TMR3H = 0;
					TMR3L = 1;
					PIR2bits.TMR3IF = 0;
					advanceMotor = 0;
					stepsDone--;
					
					//If start position is reached...
					if(stepsDone == 0)
					{
					    //Advance to next phase
					    radarPhase = RADAR_SEND;						
					}
			    }
				break;   
		}
	}
		
	BlinkUSBStatus();
	INTCONbits.GIE = 0;
    if((usb_device_state < CONFIGURED_STATE)||(UCONbits.SUSPND==1)) return;
	
	//Transmit the data if 2 pings have been collected
	if(!mUSBIntTxIsBusy() && numOfPings == 2) //user code
	{			
		counter = 0x06;	 //user code
		USBIntWrite((byte*)&dataPacket, counter); //user code
		
		numOfPings = 0;
	}

	ServiceRequests();	
}//end ProcessIO

void ServiceRequests(void)
{
  	if(USBGenRead((byte*)&dataPacket,sizeof(dataPacket)))
    {
        counter = 0;
        switch(dataPacket.CMD)
        {
            case READ_VERSION:
                //dataPacket._byte[1] is len
                dataPacket._byte[2] = MINOR_VERSION;
                dataPacket._byte[3] = MAJOR_VERSION;
                counter=0x04;
                break;

            case UPDATE_LED:
                // LED1 & LED2 are used as USB event indicators.
                if(dataPacket.led_num == 3)
                {
                    mLED_3 = dataPacket.led_status;
                    counter = 0x01;
                }//end if
                else if(dataPacket.led_num == 4)
                {
                    mLED_4 = dataPacket.led_status;
                    counter = 0x01;
                }//end if else
                break;
                
            case SET_STEPS:
                //Set the number of steps
                stepsToDo = dataPacket.led_num;
                counter = 0x01;
                
                break;
                
            case RESET:
                Reset();
                break;
                
            default:
                break;
        }//end switch()
        if(counter != 0)
        {
            if(!mUSBGenTxIsBusy())
                USBGenWrite((byte*)&dataPacket,counter);
        }//end if
    }//end if

}//end ServiceRequests

/******************************************************************************
 * Function:        void BlinkUSBStatus(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        BlinkUSBStatus turns on and off LEDs corresponding to
 *                  the USB device state.
 *
 * Note:            mLED macros can be found in io_cfg.h
 *                  usb_device_state is declared in usbmmap.c and is modified
 *                  in usbdrv.c, usbctrltrf.c, and usb9.c
 *****************************************************************************/
void BlinkUSBStatus(void)
{
    static word led_count=0;
    
    if(led_count == 0)led_count = 10000U;
    led_count--;

    #define mLED_Both_Off()         {mLED_1_Off();mLED_2_Off();}
    #define mLED_Both_On()          {mLED_1_On();mLED_2_On();}
    #define mLED_Only_1_On()        {mLED_1_On();mLED_2_Off();}
    #define mLED_Only_2_On()        {mLED_1_Off();mLED_2_On();}

    if(UCONbits.SUSPND == 1)
    {
        if(led_count==0)
        {
            mLED_1_Toggle();
            mLED_2 = mLED_1;        // Both blink at the same time
        }//end if
    }
    else
    {
        if(usb_device_state == DETACHED_STATE)
        {
            mLED_Both_Off();
            
        }
        else if(usb_device_state == ATTACHED_STATE)
        {
            mLED_Both_On();
        }
        else if(usb_device_state == POWERED_STATE)
        {
            mLED_Only_1_On();
        }
        else if(usb_device_state == DEFAULT_STATE)
        {
            mLED_Only_2_On();
        }
        else if(usb_device_state == ADDRESS_STATE)
        {
            if(led_count == 0)
            {
                mLED_1_Toggle();
                mLED_2_Off();
            }//end if
        }
        else if(usb_device_state == CONFIGURED_STATE)
        {
            if(led_count==0)
            {
                mLED_1_Toggle();
                mLED_2 = !mLED_1;       // Alternate blink                
            }//end if
        }//end if(...)
    }//end if(UCONbits.SUSPND...)

}//end BlinkUSBStatus

//User interrupt service routine
void user_interrupt(void)
{
	if(PIR2bits.TMR3IF)
	{
		mLED_3_Toggle();
//		static unsigned short long pulsecount = 0;
		TMR3H += 128; //correction factor for ~100Hz interrupt rate
//		pulsecount++;
		PIR2bits.TMR3IF = 0;
		
		advanceMotor = 1;
	}
}



void StepForward(void)
{
	stepperPos++;
	
	stepperPos = stepperPos%4;
	
	PORTAbits.RA0 = out1[stepperPos];
	PORTAbits.RA1 = out2[stepperPos];
	PORTAbits.RA2 = out3[stepperPos];
	PORTAbits.RA3 = out4[stepperPos];
}

void StepBackward(void)
{
	stepperPos--;
	
	if (stepperPos > 3)
	{
		stepperPos = 3;	
	}
	
	PORTAbits.RA0 = out1[stepperPos];
	PORTAbits.RA1 = out2[stepperPos];
	PORTAbits.RA2 = out3[stepperPos];
	PORTAbits.RA3 = out4[stepperPos];
}

/** EOF user.c ***************************************************************/
